<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Mail\BookingReceived;

class BookingController extends Controller
{
    /**
     * Display a listing of all package inquiries.
     */
    public function index()
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $bookings = Booking::with('package')->get();
        return view('admin.package_inquries', compact('bookings'));
    }

    /**
     * Remove the specified booking from storage.
     */
    public function destroy(Booking $booking)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $booking->delete();
        return redirect()->route('package_inquries')->with('success', 'Inquiry deleted successfully.');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'package_id'    => 'required|exists:tour_packages,id',
            'name'          => 'required|string|max:255',
            'email'         => 'required|email|max:255',
            'mobile'        => 'required|string|max:20',
            'num_people'    => 'required|integer|min:1',
            'expected_date' => 'required|date',
            'country'       => 'required|string|max:100',
            'notes'         => 'nullable|string',
        ]);

        // Normalize field names to match Booking model
        $bookingData = [
            'package_id'    => $data['package_id'],
            'user_name'     => $data['name'],
            'num_people'    => $data['num_people'],
            'expected_date' => $data['expected_date'],
            'country'       => $data['country'],
            'mobile'        => $data['mobile'],
            'email'         => $data['email'],
            'special_req'   => $data['notes'] ?? null,
        ];

        $booking = Booking::create($bookingData);

        // Send notification email
        try {
            $adminEmail = config('mail.from.address');
            if ($adminEmail) {
                Mail::to($adminEmail)->send(new BookingReceived($booking));
            }
        } catch (\Exception $e) {
            logger()->error('Failed to send booking notification: ' . $e->getMessage());
        }

        return redirect()->back()->with('success', 'Booking submitted. We will contact you soon.');
    }
}
