<?php


namespace App\Http\Controllers;

use App\Models\Destination;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;

class DestinationController extends Controller
{
    public function create()
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        // Ensure only admins can access
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        return view('admin.destinations_create');
    }

    public function store(Request $request)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        // Validate request
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'required|string',
            'map_url' => 'required|string', // JSON string with lat/lng
            'description' => 'nullable|string',
            'key_points' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Process key_points into array
        $keyPoints = $request->key_points
            ? array_filter(array_map('trim', explode("\n", $request->key_points)))
            : [];

        // Handle image upload
        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('destinations', 'public');
        }

        // Create destination
        Destination::create([
            'name' => $validated['name'],
            'address' => $validated['address'],
            'map_url' => $validated['map_url'],
            'description' => $validated['description'],
            'key_points' => $keyPoints,
            'image_path' => $imagePath,
        ]);

        return redirect()->route('destinations_all')->with('success', 'Destination created successfully.');
    }

    public function index()
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        // Ensure only admins can access
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        // Fetch all destinations
        $destinations = Destination::all();

        return view('admin.destinations_all', compact('destinations'));
    }

    public function edit(Destination $destination)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        // Ensure only admins can access
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        return view('admin.destinations_edit', compact('destination'));
    }

    public function update(Request $request, Destination $destination)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        // Validate request
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'required|string',
            'map_url' => 'required|string',
            'description' => 'nullable|string',
            'key_points' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Process key_points into array
        $keyPoints = $request->key_points
            ? array_filter(array_map('trim', explode("\n", $request->key_points)))
            : [];

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($destination->image_path) {
                Storage::disk('public')->delete($destination->image_path);
            }
            $destination->image_path = $request->file('image')->store('destinations', 'public');
        }

        // Update destination
        $destination->update([
            'name' => $validated['name'],
            'address' => $validated['address'],
            'map_url' => $validated['map_url'],
            'description' => $validated['description'],
            'key_points' => $keyPoints,
            'image_path' => $destination->image_path,
        ]);

        return redirect()->route('destinations_all')->with('success', 'Destination updated successfully.');
    }

    public function destroy(Destination $destination)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        // Ensure only admins can access
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        // Delete image if exists
        if ($destination->image_path) {
            Storage::disk('public')->delete($destination->image_path);
        }

        // Delete destination
        $destination->delete();

        return redirect()->route('destinations_all')->with('success', 'Destination deleted successfully.');
    }


    public function frontIndex()
    {
        // Fetch all destinations with their events eager-loaded
        $destinations = Destination::with('events')->get();

        // Extract unique countries for filtering
        $countries = $destinations->pluck('country')->unique()->sort()->values();

        return view('front.destinations', compact('destinations', 'countries'));
    }
}
