<?php

namespace App\Http\Controllers;

use App\Models\Event;
use App\Models\Destination;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class EventController extends Controller
{

    /**
     * Show the form for creating a new event.
     */
    public function create()
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $destinations = Destination::all();
        return view('admin.events_create', compact('destinations'));
    }

    /**
     * Store a newly created event in storage.
     */
    public function store(Request $request)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'destination_id' => 'required|exists:destinations,id',
        ]);

        Event::create($validated);

        return redirect()->route('events_all')->with('success', 'Event created successfully!');
    }

    /**
     * Show the form for editing an event.
     */
    public function edit($id)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $event = Event::findOrFail($id);
        $destinations = Destination::all();
        return view('admin.events_edit', compact('event', 'destinations'));
    }

    /**
     * Update the specified event in storage.
     */
    public function update(Request $request, $id)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $event = Event::findOrFail($id);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'destination_id' => 'required|exists:destinations,id',
        ]);

        $event->update($validated);

        return redirect()->route('events_all')->with('success', 'Event updated successfully!');
    }

    /**
     * Remove the specified event from storage.
     */
    public function destroy($id)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $event = Event::findOrFail($id);
        $event->delete();

        return redirect()->route('events_all')->with('success', 'Event deleted successfully!');
    }


    /**
     * Display a listing of the events.
     */
    /**
     * Display a listing of the events.
     */
    public function index()
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $events = Event::with('destination')->get();
        return view('admin.events_all', compact('events'));
    }
}
