<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Newsletter;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Mail\NewsletterCampaign;

class NewsletterController extends Controller
{
    public function store(Request $request)
    {
        // Check if cookie already exists
        if ($request->cookie('subscribed')) {
            return back()->with('error', 'You have already subscribed!');
        }

        // Validate email
        $request->validate([
            'email' => 'required|email|unique:newsletter,email',
        ]);

        // Save email
        Newsletter::create([
            'email' => $request->email,
            'subscribe_at' => now(),
        ]);

        // Set cookie (valid for 1 year)
        Cookie::queue('subscribed', true, 525600);

        return back()->with('success', 'Thank you for subscribing!');
    }


    /**
     * Display a listing of newsletter subscriptions.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $newsletters = Newsletter::all();
        return view('admin.newsletter_admin', compact('newsletters'));
    }

    /**
     * Remove the specified newsletter subscription.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $newsletter = Newsletter::findOrFail($id);
        $newsletter->delete();

        return redirect()->route('newsletter_admin')->with('success', 'Subscription removed successfully.');
    }

    /**
     * Send email campaign to all subscribers.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function sendCampaign(Request $request)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $request->validate([
            'subject' => 'required|string|max:255',
            'content' => 'required|string',
        ]);

        $subscribers = Newsletter::all();
        foreach ($subscribers as $subscriber) {
            Mail::to($subscriber->email)->send(new NewsletterCampaign($request->subject, $request->content));
        }

        return redirect()->route('newsletter_admin')->with('success', 'Email campaign sent successfully.');
    }
}
