<?php

namespace App\Http\Controllers;

use App\Models\TourPackage;
use App\Models\Category;
use App\Models\SubCategory;
use App\Models\PackageItinerary;
use App\Models\ItineraryImage;
use App\Models\Itinerary;
use Illuminate\Http\Request;
use App\Models\PackageImage;
use App\Models\Inclution;
use App\Models\Exclution;
use App\Models\Activity;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;

class PackageController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $tour = TourPackage::with(['category', 'subCategory'])->get();
        return view('admin.package.tour_packages', compact('tour'));
    }

    /**
     * Front-end packages listing with search and filters
     */
    public function frontIndex(Request $request)
    {
        $query = TourPackage::with(['category', 'subCategory', 'images']);

        if ($request->filled('search')) {
            $s = $request->input('search');
            $query->where('title', 'like', "%{$s}%")
                ->orWhere('description', 'like', "%{$s}%");
        }

        if ($request->filled('category_id')) {
            $query->where('category_id', $request->input('category_id'));
        }

        if ($request->filled('sub_category_id')) {
            $query->where('sub_category_id', $request->input('sub_category_id'));
        }

        $packages = $query->get();

        // send categories/subcategories for filter dropdowns
        $categories = \App\Models\Category::all();
        $subcategories = \App\Models\SubCategory::all();

        return view('front.packages', compact('packages', 'categories', 'subcategories'));
    }

    /**
     * Front-end single package detail by slug
     */
    public function showFront($slug)
    {

        // Eager-load related relations used by the front-end view
        $package = TourPackage::with([
            'category',
            'subCategory',
            'images',
            'itineraries.images',
            'itineraries.activities',
            'inclution',
            'exclution',
        ])->where('slug', $slug)->firstOrFail();

        $itineraries = PackageItinerary::where('package_id', $package->id)
            ->with('images')
            ->withCount('activities')
            // eager-load a small preview of activities (first 3)
            ->with(['activities' => function ($q) {
                $q->orderBy('created_at')->limit(3);
            }])
            ->orderBy('day_number')
            ->get();

        return view('front.package_detail', compact('package', 'itineraries'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $categories = Category::all();
        $subcategories = SubCategory::all();
        return view('admin.package.tour_package_create', compact('categories', 'subcategories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'category_id' => 'required|exists:categories,id',
            'sub_category_id' => 'required|exists:sub_categories,id',
        ]);

        TourPackage::create([
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'description' => $request->description,
            'price' => $request->price,
            'category_id' => $request->category_id,
            'sub_category_id' => $request->sub_category_id,
        ]);

        return redirect()->route('packages.index')->with('success', 'Tour package created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(TourPackage $package)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $categories = Category::all();
        $subcategories = SubCategory::all();
        return view('admin.package.tour_package_edit', compact('package', 'categories', 'subcategories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, TourPackage $package)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'title' => 'required|string|max:255|unique:tour_packages,slug,' . $package->id,
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'category_id' => 'required|exists:categories,id',
            'sub_category_id' => 'required|exists:sub_categories,id',
        ]);

        $package->update([
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'description' => $request->description,
            'price' => $request->price,
            'category_id' => $request->category_id,
            'sub_category_id' => $request->sub_category_id,
        ]);

        return redirect()->route('packages.index')->with('success', 'Tour package updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(TourPackage $package)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $package->delete();
        return redirect()->route('packages.index')->with('success', 'Tour package deleted successfully.');
    }

    /**
     * Display itineraries for a specific package
     */
    public function itineraries(TourPackage $package)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        // Eager-load images to ensure $itinerary->images is available in the view
        $itineraries = PackageItinerary::where('package_id', $package->id)
            ->with('images')
            ->withCount('activities')
            // eager-load a small preview of activities (first 3)
            ->with(['activities' => function ($q) {
                $q->orderBy('created_at')->limit(3);
            }])
            ->orderBy('day_number')
            ->get();
        return view('admin.package_iteneries.package_iteneries', compact('package', 'itineraries'));
    }

    /**
     * Show the form for creating a new itinerary
     */
    public function create_itineraries(TourPackage $package)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        return view('admin.package_iteneries.create_itenery', compact('package'));
    }

    /**
     * Store a new itinerary for a specific package
     */
    public function storeItineraries(Request $request, TourPackage $package)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'day_number' => 'required|integer|between:1,10',
            'title' => 'required|string|max:255',
            'map_url' => 'required|json',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'captions.*' => 'nullable|string|max:255',
        ]);

        // Create the itinerary
        $itinerary = PackageItinerary::create([
            'package_id' => $package->id,
            'day_number' => $request->day_number,
            'title' => $request->title,
            'map_url' => $request->map_url,
        ]);

        // Handle image uploads (up to 3)
        if ($request->hasFile('images')) {
            $images = $request->file('images');
            $captions = $request->input('captions', []);
            $imageCount = min(count($images), 3); // Limit to 3 images

            for ($i = 0; $i < $imageCount; $i++) {
                if ($images[$i]->isValid()) {
                    $imagePath = $images[$i]->store('itinerary_images', 'public');
                    ItineraryImage::create([
                        'package_itinerary_id' => $itinerary->id,
                        'image_path' => $imagePath,
                        'caption' => $captions[$i] ?? null,
                        'day_number' => $request->day_number,
                    ]);
                }
            }
        }

        return redirect()->route('packages.itineraries', $package->id)->with('success', 'Itinerary added successfully.');
    }

    /**
     * Show the form for editing an itinerary
     */
    public function editItinerary(TourPackage $package, PackageItinerary $itinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        return view('admin.package_iteneries.edit_itinerary', compact('package', 'itinerary'));
    }

    /**
     * Update an itinerary
     */
    public function updateItinerary(Request $request, TourPackage $package, PackageItinerary $itinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'day_number' => 'required|integer|between:1,10',
            'title' => 'required|string|max:255',
            'map_url' => 'required|json',
            'replace_images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'existing_captions.*' => 'nullable|string|max:255',
            'new_images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'new_captions.*' => 'nullable|string|max:255',
        ]);

        // Update itinerary
        $itinerary->update([
            'day_number' => $request->day_number,
            'title' => $request->title,
            'map_url' => $request->map_url,
        ]);

        // 1) Update captions for existing images (if provided)
        $existingCaptions = $request->input('existing_captions', []);
        if (!empty($existingCaptions) && is_array($existingCaptions)) {
            foreach ($existingCaptions as $imageId => $caption) {
                $img = ItineraryImage::find($imageId);
                if ($img && $img->package_itinerary_id == $itinerary->id) {
                    $img->caption = $caption;
                    $img->day_number = $request->day_number; // keep day_number in sync
                    $img->save();
                }
            }
        }

        // 2) Handle replacements for specific existing images
        if ($request->hasFile('replace_images')) {
            $replaceFiles = $request->file('replace_images');
            // replace_files is expected to be keyed by image id — be defensive about keys
            foreach ($replaceFiles as $imageId => $file) {
                // skip empty uploads
                if (!$file) continue;
                if (!is_numeric($imageId)) continue;
                $imageId = (int) $imageId;
                if ($file && $file->isValid()) {
                    $img = ItineraryImage::find($imageId);
                    if ($img && $img->package_itinerary_id == $itinerary->id) {
                        // delete old file
                        Storage::disk('public')->delete($img->image_path);
                        // store new file
                        $path = $file->store('itinerary_images', 'public');
                        $img->image_path = $path;
                        $img->day_number = $request->day_number;
                        // caption may have been updated above via existing_captions
                        $img->save();
                    }
                }
            }
        }

        // 3) Handle adding new images (append, do not remove existing ones). Limit to 3 additional at once.
        if ($request->hasFile('new_images')) {
            $newImages = $request->file('new_images');
            $newCaptions = $request->input('new_captions', []);
            $added = 0;
            // Iterate safely in case input keys are sparse; allow up to 3 new uploads
            foreach ($newImages as $idx => $file) {
                if ($added >= 3) {
                    break;
                }
                if ($file && $file->isValid()) {
                    $imagePath = $file->store('itinerary_images', 'public');
                    $caption = null;
                    if (is_array($newCaptions) && array_key_exists($idx, $newCaptions)) {
                        $caption = $newCaptions[$idx];
                    }
                    ItineraryImage::create([
                        'package_itinerary_id' => $itinerary->id,
                        'image_path' => $imagePath,
                        'caption' => $caption,
                        'day_number' => $request->day_number,
                    ]);
                    $added++;
                }
            }
        }

        return redirect()->route('packages.itineraries', $package->id)->with('success', 'Itinerary updated successfully.');
    }

    /**
     * Delete an itinerary
     */
    public function destroyItinerary(TourPackage $package, PackageItinerary $itinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        // Delete associated images
        foreach ($itinerary->images as $image) {
            Storage::disk('public')->delete($image->image_path);
            $image->delete();
        }

        $itinerary->delete();
        return redirect()->route('packages.itineraries', $package->id)->with('success', 'Itinerary deleted successfully.');
    }

    /**
     * Display sub-itineraries for a specific itinerary
     */
    public function subItineraries(TourPackage $package, PackageItinerary $itinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $subItineraries = Itinerary::where('package_itinerary_id', $itinerary->id)->orderBy('order')->get();
        return view('admin.package_iteneries.sub_itineraries', compact('package', 'itinerary', 'subItineraries'));
    }

    /**
     * Store a new sub-itinerary (list detail) for a package itinerary.
     */
    public function storeSubItinerary(Request $request, TourPackage $package, PackageItinerary $itinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'detail' => 'required|string|max:1000',
        ]);

        // Determine next order value
        $maxOrder = Itinerary::where('package_itinerary_id', $itinerary->id)->max('order');
        $nextOrder = $maxOrder ? $maxOrder + 1 : 1;

        $sub = Itinerary::create([
            'detail' => $request->input('detail'),
            'order' => $nextOrder,
            'package_itinerary_id' => $itinerary->id,
        ]);

        return response()->json(['success' => true, 'subItinerary' => $sub], 201);
    }

    /**
     * Update an existing sub-itinerary.
     */
    public function updateSubItinerary(Request $request, TourPackage $package, PackageItinerary $itinerary, Itinerary $subItinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'detail' => 'required|string|max:1000',
        ]);

        // Ensure the sub-itinerary belongs to this package itinerary
        if ($subItinerary->package_itinerary_id != $itinerary->id) {
            return response()->json(['error' => 'Mismatched itinerary'], 403);
        }

        $subItinerary->update([
            'detail' => $request->input('detail'),
        ]);

        return response()->json(['success' => true, 'subItinerary' => $subItinerary]);
    }

    /**
     * Delete a sub-itinerary.
     */
    public function destroySubItinerary(TourPackage $package, PackageItinerary $itinerary, Itinerary $subItinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        if ($subItinerary->package_itinerary_id != $itinerary->id) {
            return response()->json(['error' => 'Mismatched itinerary'], 403);
        }

        $subItinerary->delete();

        // Re-order remaining items to keep orders contiguous
        $items = Itinerary::where('package_itinerary_id', $itinerary->id)->orderBy('order')->get();
        $order = 1;
        foreach ($items as $it) {
            $it->order = $order++;
            $it->save();
        }

        return response()->json(['success' => true]);
    }

    /**
     * Reorder sub-itineraries. Expects an array of IDs in 'order' parameter.
     */
    public function reorderSubItineraries(Request $request, TourPackage $package, PackageItinerary $itinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'order' => 'required|array',
            'order.*' => 'integer|exists:itineraries,id',
        ]);

        $ids = $request->input('order');
        $order = 1;
        foreach ($ids as $id) {
            $sub = Itinerary::find($id);
            if ($sub && $sub->package_itinerary_id == $itinerary->id) {
                $sub->order = $order++;
                $sub->save();
            }
        }

        return response()->json(['success' => true]);
    }



    public function storeInclusionsExclusions(Request $request, TourPackage $package)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        // Validate the request
        $request->validate([
            'images.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'captions.*' => 'nullable|string|max:255',
        ]);

        // Handle inclusions - store as array
        if ($request->inclusions && is_array($request->inclusions)) {
            $filteredInclusions = array_filter(array_map('trim', $request->inclusions), function ($inclusion) {
                return !empty($inclusion);
            });

            if (!empty($filteredInclusions)) {
                // Delete existing inclusions for this package
                Inclution::where('package_id', $package->id)->delete();

                // Create new inclusion record with array data
                Inclution::create([
                    'package_id' => $package->id,
                    'inclution' => json_encode(array_values($filteredInclusions)),
                ]);
            }
        }

        // Handle exclusions - store as array
        if ($request->exclusions && is_array($request->exclusions)) {
            $filteredExclusions = array_filter(array_map('trim', $request->exclusions), function ($exclusion) {
                return !empty($exclusion);
            });

            if (!empty($filteredExclusions)) {
                // Delete existing exclusions for this package
                Exclution::where('package_id', $package->id)->delete();

                // Create new exclusion record with array data
                Exclution::create([
                    'package_id' => $package->id,
                    'exclution' => json_encode(array_values($filteredExclusions)),
                ]);
            }
        }

        // Handle images (up to 4 images)
        if ($request->hasFile('images')) {
            $images = array_slice($request->file('images'), 0, 4); // Limit to 4 images
            $captions = $request->captions ?? [];

            // Delete existing images for this package
            $existingImages = PackageImage::where('package_id', $package->id)->get();
            foreach ($existingImages as $existingImage) {
                // Delete physical file
                if (file_exists(public_path($existingImage->image_path))) {
                    unlink(public_path($existingImage->image_path));
                }
            }

            PackageImage::where('package_id', $package->id)->delete();

            // Store new images
            foreach ($images as $index => $image) {
                $imageName = time() . '' . $index . '' . $image->getClientOriginalName();
                $imagePath = 'uploads/packages/' . $imageName;

                // Create uploads directory if it doesn't exist
                if (!file_exists(public_path('uploads/packages'))) {
                    mkdir(public_path('uploads/packages'), 0755, true);
                }

                // Move uploaded file
                $image->move(public_path('uploads/packages'), $imageName);

                // Save to database
                PackageImage::create([
                    'package_id' => $package->id,
                    'image_path' => $imagePath,
                    'caption' => isset($captions[$index]) ? trim($captions[$index]) : null,
                ]);
            }
        }

        return redirect()->route('packages.index')
            ->with('success', 'Inclusions, exclusions, and images saved successfully.');
    }

    public function images(TourPackage $package)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        return view('admin.package.tour_package_images', compact('package'));
    }

    /**
     * Show the page to manage inclusions, exclusions and package images
     */
    public function package_inclution_exlution_details(TourPackage $package)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        // Load related data: images, inclutions, exclutions
        $images = PackageImage::where('package_id', $package->id)->get();
        $inclution = Inclution::where('package_id', $package->id)->first();
        $exclution = Exclution::where('package_id', $package->id)->first();

        // Decode stored JSON arrays if available
        $inclusions = [];
        $exclusions = [];
        if ($inclution && $inclution->inclution) {
            $decoded = json_decode($inclution->inclution, true);
            if (is_array($decoded)) $inclusions = $decoded;
        }
        if ($exclution && $exclution->exclution) {
            $decoded = json_decode($exclution->exclution, true);
            if (is_array($decoded)) $exclusions = $decoded;
        }

        return view('admin.package.package_inclution_exlution', compact('package', 'images', 'inclusions', 'exclusions'));
    }

    /**
     * Update a single package image (caption and/or replace file)
     */
    public function updatePackageImage(Request $request, TourPackage $package, PackageImage $image)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        // ensure image belongs to package
        if ($image->package_id != $package->id) {
            return redirect()->back()->with('error', 'Image does not belong to this package.');
        }

        $request->validate([
            'caption' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Replace file if provided
        if ($request->hasFile('image')) {
            // delete old file
            if ($image->image_path && file_exists(public_path($image->image_path))) {
                unlink(public_path($image->image_path));
            }

            $file = $request->file('image');
            $imageName = time() . '_' . $file->getClientOriginalName();
            if (!file_exists(public_path('uploads/packages'))) {
                mkdir(public_path('uploads/packages'), 0755, true);
            }
            $file->move(public_path('uploads/packages'), $imageName);
            $image->image_path = 'uploads/packages/' . $imageName;
        }

        $image->caption = $request->input('caption');
        $image->save();

        return redirect()->route('package_inclution_exlution_details', $package->id)->with('success', 'Image updated.');
    }

    /**
     * Delete a single package image
     */
    public function destroyPackageImage(TourPackage $package, PackageImage $image)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        if ($image->package_id != $package->id) {
            return redirect()->back()->with('error', 'Image does not belong to this package.');
        }

        if ($image->image_path && file_exists(public_path($image->image_path))) {
            unlink(public_path($image->image_path));
        }

        $image->delete();

        return redirect()->route('package_inclution_exlution_details', $package->id)->with('success', 'Image deleted.');
    }

    /**
     * Update an inclusion item by index
     */
    public function updateInclusion(Request $request, TourPackage $package, $index)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate(['value' => 'required|string|max:255']);

        $incl = Inclution::where('package_id', $package->id)->first();
        if (!$incl) return redirect()->back()->with('error', 'No inclusions found.');

        $arr = json_decode($incl->inclution, true) ?: [];
        if (!isset($arr[$index])) return redirect()->back()->with('error', 'Inclusion not found.');

        $arr[$index] = $request->input('value');
        $incl->inclution = json_encode(array_values($arr));
        $incl->save();

        return redirect()->route('package_inclution_exlution_details', $package->id)->with('success', 'Inclusion updated.');
    }

    /**
     * Delete an inclusion by index
     */
    public function destroyInclusion(TourPackage $package, $index)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $incl = Inclution::where('package_id', $package->id)->first();
        if (!$incl) return redirect()->back()->with('error', 'No inclusions found.');

        $arr = json_decode($incl->inclution, true) ?: [];
        if (!isset($arr[$index])) return redirect()->back()->with('error', 'Inclusion not found.');

        array_splice($arr, $index, 1);
        $incl->inclution = json_encode(array_values($arr));
        $incl->save();

        return redirect()->route('package_inclution_exlution_details', $package->id)->with('success', 'Inclusion removed.');
    }

    /**
     * Update an exclusion item by index
     */
    public function updateExclusion(Request $request, TourPackage $package, $index)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate(['value' => 'required|string|max:255']);

        $ex = Exclution::where('package_id', $package->id)->first();
        if (!$ex) return redirect()->back()->with('error', 'No exclusions found.');

        $arr = json_decode($ex->exclution, true) ?: [];
        if (!isset($arr[$index])) return redirect()->back()->with('error', 'Exclusion not found.');

        $arr[$index] = $request->input('value');
        $ex->exclution = json_encode(array_values($arr));
        $ex->save();

        return redirect()->route('package_inclution_exlution_details', $package->id)->with('success', 'Exclusion updated.');
    }

    /**
     * Delete an exclusion by index
     */
    public function destroyExclusion(TourPackage $package, $index)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $ex = Exclution::where('package_id', $package->id)->first();
        if (!$ex) return redirect()->back()->with('error', 'No exclusions found.');

        $arr = json_decode($ex->exclution, true) ?: [];
        if (!isset($arr[$index])) return redirect()->back()->with('error', 'Exclusion not found.');

        array_splice($arr, $index, 1);
        $ex->exclution = json_encode(array_values($arr));
        $ex->save();

        return redirect()->route('package_inclution_exlution_details', $package->id)->with('success', 'Exclusion removed.');
    }

    /**
     * Display activities for a specific itinerary
     */
    public function activities(TourPackage $package, PackageItinerary $itinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $activities = Activity::where('itinerary_id', $itinerary->id)
            ->orderBy('created_at')
            ->get();

        return view('admin.activities.activities', compact('package', 'itinerary', 'activities'));
    }

    /**
     * Store a new activity for a specific itinerary
     */
    public function storeActivity(Request $request, TourPackage $package, PackageItinerary $itinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $imagePath = null;

        // Handle image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $imagePath = 'uploads/activities/' . $imageName;

            // Create uploads directory if it doesn't exist
            if (!file_exists(public_path('uploads/activities'))) {
                mkdir(public_path('uploads/activities'), 0755, true);
            }

            $image->move(public_path('uploads/activities'), $imageName);
        }

        Activity::create([
            'itinerary_id' => $itinerary->id,
            'name' => $request->name,
            'description' => $request->description,
            'image_path' => $imagePath,
        ]);

        return redirect()->route('packages.itineraries.activities', [$package->id, $itinerary->id])
            ->with('success', 'Activity created successfully.');
    }

    /**
     * Update an existing activity
     */
    public function updateActivity(Request $request, TourPackage $package, PackageItinerary $itinerary, Activity $activity)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Ensure the activity belongs to this itinerary
        if ($activity->itinerary_id != $itinerary->id) {
            return redirect()->back()->with('error', 'Activity does not belong to this itinerary.');
        }

        $imagePath = $activity->image_path;

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($activity->image_path && file_exists(public_path($activity->image_path))) {
                unlink(public_path($activity->image_path));
            }

            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $imagePath = 'uploads/activities/' . $imageName;

            // Create uploads directory if it doesn't exist
            if (!file_exists(public_path('uploads/activities'))) {
                mkdir(public_path('uploads/activities'), 0755, true);
            }

            $image->move(public_path('uploads/activities'), $imageName);
        }

        $activity->update([
            'name' => $request->name,
            'description' => $request->description,
            'image_path' => $imagePath,
        ]);

        return redirect()->route('packages.itineraries.activities', [$package->id, $itinerary->id])
            ->with('success', 'Activity updated successfully.');
    }

    /**
     * Delete an activity
     */
    public function destroyActivity(TourPackage $package, PackageItinerary $itinerary, Activity $activity)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        // Ensure the activity belongs to this itinerary
        if ($activity->itinerary_id != $itinerary->id) {
            return redirect()->back()->with('error', 'Activity does not belong to this itinerary.');
        }

        // Delete image if exists
        if ($activity->image_path && file_exists(public_path($activity->image_path))) {
            unlink(public_path($activity->image_path));
        }

        $activity->delete();

        return redirect()->route('packages.itineraries.activities', [$package->id, $itinerary->id])
            ->with('success', 'Activity deleted successfully.');
    }

    /**
     * Reorder activities
     */
    public function reorderActivities(Request $request, TourPackage $package, PackageItinerary $itinerary)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'order' => 'required|array',
            'order.*' => 'integer|exists:activities,id',
        ]);

        $ids = $request->input('order');
        $order = 1;

        foreach ($ids as $id) {
            $activity = Activity::find($id);
            if ($activity && $activity->itinerary_id == $itinerary->id) {
                $activity->update(['order' => $order++]);
            }
        }

        return response()->json(['success' => true]);
    }

    /**
     * Return a single activity as JSON for AJAX requests.
     */
    public function getActivity(Activity $activity)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        if (!$activity) {
            return response()->json(['error' => 'Activity not found'], 404);
        }

        // Ensure image path is a full URL if present
        $activityData = $activity->toArray();
        if (!empty($activity->image_path)) {
            $activityData['image_path'] = asset($activity->image_path);
        } else {
            $activityData['image_path'] = null;
        }

        return response()->json($activityData);
    }
}
