<?php

namespace App\Http\Controllers;

use App\Models\Review;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ReviewController extends Controller
{
    /**
     * Display a listing of all reviews.
     */
    public function index()
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $reviews = Review::with('package')->get();
        return view('admin.reviews_all', compact('reviews'));
    }

    /**
     * Update the status of a review (Approve/Reject/Pending).
     */
    public function updateStatus(Request $request, Review $review)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'status' => 'required|in:approved,rejected,pending',
        ]);

        $review->update([
            'status' => $request->status,
        ]);

        return redirect()->route('reviews_all')->with('success', 'Review status updated successfully.');
    }

    /**
     * Store a front-end submitted review.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'package_id' => 'required|exists:tour_packages,id',
            'user_name' => 'required|string|max:255',
            'ratings' => 'required|integer|min:1|max:5',
            'comments' => 'nullable|string',
        ]);

        $data['status'] = 'pending';

        Review::create($data);

        return redirect()->back()->with('success', 'Thank you! Your review has been submitted and is pending approval.');
    }

    /**
     * Remove the specified review from storage.
     */
    public function destroy(Review $review)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        $review->delete();
        return redirect()->route('reviews_all')->with('success', 'Review deleted successfully.');
    }
}
