<?php

namespace App\Http\Controllers;

use App\Models\VideoGallery;
use App\Models\GalleryCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class VideoGalleryController extends Controller
{

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $videos = VideoGallery::with('category')->orderBy('created_at', 'desc')->get();
        $categories = GalleryCategory::all();
        return view('admin.video_gallery_admin', compact('videos', 'categories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $categories = GalleryCategory::all();
        return view('admin.video_gallery_create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'required|exists:gallery_categories,id',
            'video_type' => 'required|in:upload,youtube',
            'video_file' => 'required_if:video_type,upload|mimes:mp4,avi,mov,wmv,flv|max:51200', // 50MB max
            'video_url' => 'required_if:video_type,youtube|url',
        ]);

        $videoUrl = null;

        if ($request->video_type === 'upload' && $request->hasFile('video_file')) {
            // Handle video file upload
            $videoPath = $request->file('video_file')->store('gallery_videos', 'public');
            $videoUrl = $videoPath;
        } elseif ($request->video_type === 'youtube') {
            // Handle YouTube URL
            $videoUrl = $this->convertYouTubeUrl($request->video_url);
        }

        VideoGallery::create([
            'title' => $request->title,
            'category_id' => $request->category_id,
            'video_url' => $videoUrl,
        ]);

        return redirect()->route('video-gallery.index')
            ->with('success', 'Video added to gallery successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $videoGallery = VideoGallery::findOrFail($id);
        $categories = GalleryCategory::all();
        return view('admin.video_gallery_edit', compact('videoGallery', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $videoGallery = VideoGallery::findOrFail($id);

        $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'required|exists:gallery_categories,id',
            'video_type' => 'required|in:upload,youtube',
            'video_file' => 'nullable|mimes:mp4,avi,mov,wmv,flv|max:51200', // 50MB max
            'video_url' => 'required_if:video_type,youtube|url',
        ]);

        $data = [
            'title' => $request->title,
            'category_id' => $request->category_id,
        ];

        // Handle video update
        if ($request->video_type === 'upload' && $request->hasFile('video_file')) {
            // Delete old video file if it exists and is not a YouTube URL
            if ($videoGallery->video_url && !$this->isYouTubeUrl($videoGallery->video_url)) {
                Storage::disk('public')->delete($videoGallery->video_url);
            }

            $data['video_url'] = $request->file('video_file')->store('gallery_videos', 'public');
        } elseif ($request->video_type === 'youtube') {
            // Delete old video file if it exists and is not a YouTube URL
            if ($videoGallery->video_url && !$this->isYouTubeUrl($videoGallery->video_url)) {
                Storage::disk('public')->delete($videoGallery->video_url);
            }

            $data['video_url'] = $this->convertYouTubeUrl($request->video_url);
        }

        $videoGallery->update($data);

        return redirect()->route('video-gallery.index')
            ->with('success', 'Video updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        if (Auth::check() && Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $videoGallery = VideoGallery::findOrFail($id);

        // Delete video file from storage if it's not a YouTube URL
        if ($videoGallery->video_url && !$this->isYouTubeUrl($videoGallery->video_url)) {
            Storage::disk('public')->delete($videoGallery->video_url);
        }

        $videoGallery->delete();

        return redirect()->route('video-gallery.index')
            ->with('success', 'Video deleted successfully.');
    }

    /**
     * Convert YouTube URL to embed format
     */
    private function convertYouTubeUrl($url)
    {
        if (preg_match('/(?:youtube\.com\/watch\?v=|youtu\.be\/)([a-zA-Z0-9_-]+)/', $url, $matches)) {
            return 'https://www.youtube.com/embed/' . $matches[1];
        }
        return $url;
    }

    /**
     * Check if URL is a YouTube URL
     */
    private function isYouTubeUrl($url)
    {
        return strpos($url, 'youtube.com') !== false || strpos($url, 'youtu.be') !== false;
    }

    public function showGallery()
    {
        // Fetch all categories
        $categories = GalleryCategory::all();

        // Fetch all videos with their associated categories, ordered by creation date
        $videos = VideoGallery::with('category')->orderBy('created_at', 'desc')->get();

        return view('front.video_gallery', compact('videos', 'categories'));
    }
}
